% fpi_compare_accuracy.m
% 
% accuracy checks of the FiPIt model
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %

% -- Clear Workspace -- %
clear;
clc;
close all;

tic;

% -- Take Symbolic Derivatives -- %
[f, Pmat, vars, THETA, steady] = bfor_init_vars_equilibriums_fipit();
[derivs] = bfor_endogenous_ms_solver_derivatives_analytic(f, Pmat, vars);

% -- System Dimension -- %
ny = 16;
nx = 5;
ne = 3;
ns = 2;
nY = 21;
obsindex  = 1:21;   

% -- Parameters -- %
p01_0       = 0.95;
p10_0       = 0.95;
ggamma00    = log(p01_0/(1-p01_0));
ggamma10    = log(p10_0/(1-p10_0));
ggamma01    = 50;
ggamma11    = 50;
ssigma      = 2.0;
oomega      = 1.8461;
bbeta       = 0.92;
iiota       = 2.75;
pphi        = 0.2579;
ddelta      = 0.08800443;
aalpha      = 0.5924043;
eeta        = 0.102229;
ggamma      = 1-aalpha-eeta;
ttau        = 0.168615385890;
astar       = 6.982;
kkappa      = 0.2;
rbar        = 1.0857;
pbar        = 1.0278;
abar        = 1;
rrhor       = 0.5551;
rrhop       = 0.7374;
rrhoa       = 0.5551;
ssigmaa     = 0.0100185;
ssigmap     = 0.023235697402196;
ssigmar     = 0.0170303;
ssigmaar    = -0.00488483;
varphi_1    = 0;
varphi_2    = 1;
nu_1        = 0;
nu_2        = 1;
% Paramter Vector
nTHETA   = eval(THETA);

% -- Model Solution -- %
[xss, yss, sss, Pss, H, G, H2, G2, retcode] = ...
            bfor_endogenous_ms_solver_main(f, Pmat, vars, derivs, steady, THETA, nTHETA);

% -- Initialization -- %
burn = 1;       % 1e3
sims = 10;      % 1e5
nint = 10;      % 1e5

P       = 0.5*ones(2,2);
xsim    = NaN(nx,burn+sims);
xsim1   = NaN(nx,burn+sims);
xsim2   = NaN(nx,burn+sims);
ysim    = NaN(ny,burn+sims);
ysim1   = NaN(ny,burn+sims);
ysim2   = NaN(ny,burn+sims);
regimes = NaN(burn+sims,1);
s       = 1;
eps     = randn(ne,burn+sims);
EEE     = zeros(2,burn+sims);

xsim1(:,1) = 0;
xsim2(:,1) = 0;
ysim1(:,1) = 0;
ysim2(:,1) = 0;
regimes(1) = s;



% -- Simulations and EEE Calculation -- %
for tt = 2:burn+sims
    if floor(tt/100) == tt/100
        disp(['Iteration = ' num2str(tt)])
        toc
        tic
    end
        
    % -- Simulate Regime -- %
    s = (rand > P(s,1)) + 1;
    regimes(tt) = s;
    
    % -- Predetermined Variables -- %
    xsim1(:,tt) = H(:,:,s)*[xsim1(:,tt-1);eps(:,tt);1];    
    xsim2(:,tt) = H(:,:,s)*[xsim2(:,tt-1);zeros(ne,1);0] + ...
        0.5*H2(:,:,s)*kron([xsim1(:,tt-1);eps(:,tt);1],[xsim1(:,tt-1);eps(:,tt);1]);
    xsim(:,tt) = xss + xsim1(:,tt) + xsim2(:,tt);
    
    % -- Nonpredetermined Variables -- %
    ysim1(:,tt) = G(:,:,s)*[xsim1(:,tt-1);eps(:,tt);1];
    ysim2(:,tt) = G(:,:,s)*[xsim2(:,tt-1);zeros(ne,1);0] + ...
        0.5*G2(:,:,s)*kron([xsim1(:,tt-1);eps(:,tt);1],[xsim1(:,tt-1);eps(:,tt);1]);
    ysim(:,tt) = yss + ysim1(:,tt) + ysim2(:,tt);
    
    % -- Transition Matrix -- %
    bstar       = ysim(8,tt);
    mu          = ysim(6,tt);
	P(1,1)      = 1 - exp(-ggamma01*bstar)/(1+exp(-ggamma01*bstar));
	P(1,2)      = exp(-ggamma01*bstar)/(1+exp(-ggamma01*bstar));
	P(2,1)      = exp(-ggamma11*mu)/(1+exp(-ggamma11*mu));
	P(2,2)      = 1 - exp(-ggamma11*mu)/(1+exp(-ggamma11*mu));
		
    c       = exp(ysim(1,tt));
    l       = exp(ysim(2,tt));
    q       = exp(ysim(4,tt));
    lambda  = exp(ysim(7,tt));
    r       = exp(ysim(10,tt));
    
    expect  = zeros(2,1);
    for ii = 1:nint
        epsp = randn(ne,1);
        for sp = 1:2
            yp1         = G(:,:,sp)*[xsim1(:,tt);epsp;1];
            yp2         = G(:,:,sp)*[xsim2(:,tt);zeros(ne,1);0] + ...
                            0.5*G2(:,:,sp)*kron([xsim1(:,tt);epsp;1],[xsim1(:,tt);epsp;1]);
            yp          = yss + yp1 + yp2;
            
            % expectations 
            cp          = exp(yp(1));
            lp          = exp(yp(2));
            qp          = exp(yp(4));
            lambdap     = exp(yp(7));
            dp          = exp(yp(9));
            expect(1)   = expect(1) + P(s,sp)*lambdap;
            expect(2)   = expect(2) + P(s,sp)*lambdap*(dp+qp);
        end
    end
    EEE(1,tt) = 1 - (((1+ttau)*(r*bbeta*expect(1)/nint + mu))^(-1/ssigma)+l^oomega/oomega)/c;
    EEE(2,tt) = 1 - (((1+ttau)*(1/q*bbeta*expect(2)/nint)+mu*kkappa)^(-1/ssigma)+l^oomega/oomega)/c;
    
end

% -- Table -- %
table([1 3],1) = log10(mean(abs(EEE(:,burn+1:burn+sims)),2));
table([2 4],1) = log10(max(abs(EEE(:,burn+1:burn+sims)),[],2));

disp(table)